<?php
/**
* phpGedView Research Assistant Tool - ra_ViewTasks
*
* phpGedView: Genealogy Viewer
* Copyright (C) 2002 to 2024 PGV Development Team.  All rights reserved.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* @package PhpGedView
* @subpackage Research_Assistant
* @version $Id: ra_ViewTask.php 7377 2024-10-18 13:50:13Z canajun2eh $:
* @author Jason Porter
* @author Wade Lasson
* @author Brandon Gagnon
* @author Brian Kramer
* @author Julian Gautier
*/

if (!defined('PGV_PHPGEDVIEW')) {
	header('HTTP/1.0 403 Forbidden');
	exit;
}

require_once PGV_ROOT.'includes/classes/class_person.php';
global $pgv_lang, $TBLPREFIX;

/**
 * GETS the TITLE of the task with the given taskid
 *
 * @return mixed title of the task
 */
function getTitle(){
	global $TBLPREFIX;

	return
		PGV_DB::prepare("SELECT t_title FROM {$TBLPREFIX}tasks WHERE t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOne();
}

/**
 * GETS the DATES of the task with the given taskid
 *
 * @return mixed dates of the task
 */
function getDates(){
	global $TBLPREFIX;

	$s_date=
		PGV_DB::prepare("SELECT t_startdate FROM {$TBLPREFIX}tasks WHERE t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOne();
	$e_date=
		PGV_DB::prepare("SELECT t_enddate FROM {$TBLPREFIX}tasks WHERE t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOne();

	// Display either the starting and ending date or just the ending date
	if (empty($e_date)) {
		return "opened: $s_date";
	} else {
		return "$s_date - $e_date";
	}
}

/**
 * GETS the Task Status based on the dates with the given taskid
 *
 * @return mixed dates of the task
 */
function getTaskStatus(){
	global $TBLPREFIX, $pgv_lang;

	$e_date=
		PGV_DB::prepare("SELECT t_enddate FROM {$TBLPREFIX}tasks WHERE t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOne();

	// Display either the starting and ending date or just the ending date
	if (empty($e_date)) {
		$status = ' ('.$pgv_lang["ra_incomplete"].')';
	} else {
		$status = ' ('.$pgv_lang["ra_complete"].')';
	}
	return $status;
}

/**
 * GETS the FOLDER of the task with the given taskid
 *
 * @return mixed list of available folders
 */
function getFolder() {
	global $TBLPREFIX;

	$folderid =
		PGV_DB::prepare("SELECT t_fr_id FROM {$TBLPREFIX}tasks WHERE t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOne();
	$rows=
		PGV_DB::prepare("SELECT fr_name, fr_id FROM {$TBLPREFIX}folders")
		->fetchAll();

	$out="";
	foreach ($rows as $row) {
		if ($row->fr_id == $folderid) {
			$out .= PrintReady($row->fr_name);
		}
	}
	return $out;
}

/**
 * GETS all PEOPLE associated with the task given taskid
 *
 * @return mixed people associated with the task
 */
function getPeople(){
	global $TBLPREFIX;

	$ids=
		PGV_DB::prepare("SELECT it_i_id FROM {$TBLPREFIX}individualtask WHERE it_t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOneColumn();

	$out="";
	foreach ($ids as $id) {
		$person=Person::getInstance($id);
		if ($person) {
			$indi = $person->getXref();
			$out.='<a href="'.$person->getLinkUrl().'">'.PrintReady($person->getFullName()." (".$indi.")").'</a><br />';
		}
	}

	return $out;
}

/**
 * GETS the USERNAME of the task with the given taskid
 *
 * @return string indicating the full users name followed by system username in brackets
 */
function gettUsername(){
	global $TBLPREFIX;

	$t_user_id =
		PGV_DB::prepare("SELECT t_username FROM {$TBLPREFIX}tasks WHERE t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOne();

	$t_user_name = "";
	foreach(get_all_users() as $user_id=>$user_name) {
		if ($t_user_id == $user_name) $t_user_name = getUserFullName($user_id);
	}
	return ($t_user_name."&nbsp;&nbsp;(".$t_user_id.")");
}

/**
 * GETS the DESCRIPTION of the task with the given taskid
 *
 * @return mixed description of the task
 */
function getDescription(){
	global $TBLPREFIX;

	return
		PGV_DB::prepare("SELECT t_description FROM {$TBLPREFIX}tasks WHERE t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOne();
}

/**
 * GETS the results of the task with the given taskid
 *
 * @return mixed description of the task
 */
function getResults(){
	global $TBLPREFIX;

	return
		PGV_DB::prepare("SELECT t_results FROM {$TBLPREFIX}tasks WHERE t_id=?")
		->execute(array($_REQUEST['taskid']))
		->fetchOne();
}

/**
 * GETS all SOURCES associated with the task given taskid
 *
 * @return sources associated with the task
 */
function getSources(){
	global $TBLPREFIX;

	$rows=
		PGV_DB::prepare("SELECT s_name, s_id FROM {$TBLPREFIX}sources, {$TBLPREFIX}tasksource WHERE s_file=? AND ts_s_id=s_id AND ts_t_id=?")
		->execute(array(PGV_GED_ID, $_REQUEST['taskid']))
		->fetchAll();

	$sources = array();
	foreach ($rows as $row) {
		$sources[$row->s_id]=$row->s_name;
	}
	return $sources;
}

/**
 * GETS all COMMENTS associated with the task
 *
 * @return mixed comments associated with the task
 */
function getComments(){
	global $TBLPREFIX, $pgv_lang;

	$rows=
		PGV_DB::prepare("SELECT c_u_username, c_body, c_datetime, c_id FROM 
		{$TBLPREFIX}comments WHERE c_t_id='" . $_REQUEST["taskid"] . "' ORDER BY c_datetime DESC")
		->fetchAll();

	$out='';
	foreach ($rows as $row) {
		$date=timestamp_to_gedcom_date($row->c_datetime);
		$out .= '<div class="blockcontent"><div class="person_box" id="comment1"><span class="news_title">' .
			$row->c_u_username .  // INSERT username
			'</span><br /><span class="news_date">' .
			$date->Display(false).' - '. date("g:i:s A",(int)$row->c_datetime). // INSERT datetime
			'</span><br /><br />' .
			nl2br($row->c_body) . // INSERT body
			'<hr size="1" />';

		if(PGV_USER_IS_ADMIN || PGV_USER_NAME==$row->c_u_username){
			// If permitted then display the edit and delete comment options

			// If edit then on no new comment, edit window closes and just continue
			// with current window.  If edit and success adding a new comment then
			// window closes and href="javascript:;" refreshes the window effectively 
			// updating the current viewtask window.
			// If delete comment then confirm first then delete if confirmed, but URL
			// has been modified so causes and issue.
			$out .= '<a href="javascript:;" onclick="editcomment('.$row->c_id.')">'.$pgv_lang["ra_edit"].'</a>';
			$out .= ' | ';
			$out .= '<a href="" onclick="confirm_deletecomment_taskv(\''.$pgv_lang["ra_comment_delete_check"].'\', '.$row->c_id.', '.$_REQUEST["taskid"].'); return false;">'.$pgv_lang["ra_delete"].'</a>';
		}
		$out .= '<br /></div></div><br/>';
	}
	return $out;
}


if (isset($_REQUEST['delete']) && !empty($_REQUEST['delete'])){
	// TODO: Verify user
	PGV_DB::prepare("DELETE FROM {$TBLPREFIX}comments WHERE c_id=?")
		->execute(array($_REQUEST['delete']));
	print "<script type=\"text/javascript\" language=\"JavaScript\"> window.location = module.php?mod=research_assistant&action=viewtask&taskid=".$_REQUEST['taskid']."</script>\n"; 
}
?>

<!--JAVASCRIPT-->
<script language="JavaScript" type="text/javascript"><!--
	function showchanges() {
		window.location = 'module.php?mod=research_assistant&action=edittask&taskid=<?php print $_REQUEST['taskid']; ?>';
	}
	function editcomment(commentid) {
	window.open('module.php?mod=research_assistant&action=editcomment&taskid=<?php print $_REQUEST['taskid']; ?>&commentid='+commentid, '', 'top=50,left=50,width=500,height=350,resizable=1,scrollbars=1');
	}
	function confirm_deletecomment2(text, commentid) {
		if (confirm(text)) {
			window.location = 'module.php?mod=research_assistant&action=viewtask&delete='+commentid+'&taskid=<?php print $_REQUEST['taskid']; ?>';
		}
	}
//-->
</script>
<!--BEGIN VIEW TASK FORM-->
<form action="module.php?mod=research_assistant&amp;action=updatetask&amp;taskid=<?php print $_REQUEST['taskid'] ?>" method="post">
	<input type="hidden" name="mod" value="research_assistant" />
	<input type="hidden" name="action" value="updatetask" />
	<input type="hidden" name="taskid" value="<?php print $_REQUEST['taskid'] ?>" />
	<table class="list_table" align="center" border="0" width="40%" max-width="100%">
<!--HEADING-->
	<tr>
	<th colspan="4" align="right" class="topbottombar">
	<h2>
		<?php  print_help_link("ra_task_view_help", "qm", '', false, false); print $pgv_lang["ra_view_task"]; ?>
	</h2>
	</th>
	</tr>
<!--TITLE - force min space for desc boxes with 15%, to help the entire table alignment -->
	<tr>
	<td class="descriptionbox" width="15%"> 
		<?php print $pgv_lang["ra_title"]; ?>
	</td>
	<td class="optionbox">
	<?php
		// get title, given taskid
		print PrintReady(getTitle().getTaskStatus());
	?>
	</td>
	</tr>
<!--FOLDER-->
	<tr>
	<td class="descriptionbox" >
		<?php print $pgv_lang["ra_Folder"]; ?>
	</td>
	<td class="optionbox" >
		<?php
			print PrintReady(nl2br(getFolder()));
		?>
	</td>
	</tr>
<!--ASSIGNED TO-->
	<tr>
	<td class="descriptionbox" >
		<?php print $pgv_lang["ra_assigned"]; ?>
	</td>
	<td class="optionbox" >
		<?php
			print PrintReady(nl2br(gettUsername()));
		?>
	</td>
	</tr>
<!--DESCRIPTION-->
	<tr>
	<td class="descriptionbox" >
		<?php print $pgv_lang["ra_description"]; ?>
	</td>
	<td class="optionbox wrap" >
		<?php
			print (nl2br(getDescription()));
		?>
	</td>
	</tr>
<!--SOURCES-->
	<tr>
	<td class="descriptionbox">
		<?php print $pgv_lang["ra_source"]; ?>
	</td>
	<td class="optionbox" >
		<?php
			$sources = getSources();
			$sval = '';
			foreach($sources as $sid=>$source) {
				$sval .= ';'.$sid;
				print '<a id="link_'.$sid.'" href="source.php?sid='.$sid.'">'.$source.'</a>';
			}
		?>
	</td>
	</tr>
<!--PEOPLE-->
	<tr>
	<td class="descriptionbox">
		<?php print $pgv_lang["ra_people"]; ?>
	</td>
	<td class="optionbox" >
		<?php
			// Get people, given taskid
			print getPeople();
		?>
	</td>
	</tr>
<!--TABLE Footer - EDIT BUTTON-->
	<tr>
	<td colspan="4" align="right" class="topbottombar">
		<?php
		if(PGV_USER_IS_ADMIN || PGV_USER_NAME==gettUsername()){
			// If permitted then display the edit task button
		?>
			<input type="button" value="<?php print $pgv_lang["ra_edit_task"]; ?>"
			onclick="window.location.href='module.php?mod=research_assistant&amp;action=edittask&amp;taskid=<?php print $_REQUEST['taskid']; ?>'"/>
		<?php
		} else print '<br />';
		?>
	</td>
	</tr>

<!-- Print an empty row break between the Task table and the Comments table -->
	<tr>
	<td colspan="4">
		<br />
	</td>
	</tr>
<!--HEADING-->
	<tr>
	<td colspan="4" align="right" class="topbottombar">
	<h3>
		<?php print_help_link("ra_comments_help", "qm", '', false, false); print $pgv_lang["ra_comments"]; ?>
	</h3>
	</td>
	</tr>
<!--COMMENT SECTION-->
	<tr>
	<td class="optionbox" colspan="4">
	<div id="gedcom_news" class="person_box">
		<table class="blockheader" cellspacing="0" cellpadding="0">
			<tr>
				<td class="blockh1" >&nbsp;</td>
				<td class="blockh2" >
					<div class="blockhc"></div>
				</td>
				<td class="blockh3">&nbsp;</td>
			</tr>
		</table>
		<?php print getComments(); ?>
	</div>
	</td>
	</tr>
<!--END COMMENT SECTION-->
	<tr>
	<td class="topbottombar" colspan="4">
		<input type="button" value="<?php print $pgv_lang["ra_add_new_comment"]; ?>" name="Add New Comment" onclick="window.open('module.php?mod=research_assistant&action=editcomment&taskid='+<?php print $_REQUEST['taskid']; ?>, '',
		'top=50,left=50,width=500,height=350,resizable=1,scrollbars=1');">
	</td>
	</tr>
	</table>
	</form>

<!--END VIEW TASK FORM-->
